
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "rigup.h"
#include "commands.h"


static int usage()
{
  puts("Usage: rigup license KEYFILE OPTIONS [OPTIONS...]");
  puts("");
  puts("Create one or more new licenses using keys and serial");
  puts("number stored in a keyfile. The license option is given");
  puts("with 4 characters or as a hexadecimal number.");
  puts("");
  puts("    KEYFILE     Keyfile containing keys and serial number.");
  puts("    OPTIONS     License options. Either four characters");
  puts("                out of: ABCDEFGHJKLMNPQRSTUVWXYZ23456789");
  puts("                or hexadecimal number in the form of 0x12345.");
  puts("                Multiple options can be given to create");
  puts("                several licenses at once.");
  puts("");
  return 0;
}


int main_createlicense(int argc, char *argv[])
{
  if (argc < 3)
    return usage();

  const char *keyfilename = argv[1];

  KeyData *keydata = LoadKeyData(keyfilename);
  if (!keydata)
  {
    fprintf(stderr, "Loading keyfile '%s' failed.\n", keyfilename);
    return 1;
  }

  for (int i = 2; i <argc; ++ i)
  {
    const char *options = argv[i];

    uint32_t optionBits;
    if (strlen(options) == 4)
    {
      optionBits = DecodeOptionBits(options);
    }
    else if (strlen(options) == 7 && strncmp(options, "0x", 2) == 0)
    {
      int matched = sscanf(options, "%x", &optionBits);
      if (matched != 1)
      {
	fprintf(stderr, "Parsing hexadecimal option bits failed.\n");
	return 1;
      }
    }
    else
    {
      fprintf(stderr, "Unknown option bits format,\n");
      return 1;
    }

    char *licenseCode = CreateLicenseCode(optionBits, keydata);
    char *pretty = PrettyLicenseCode(licenseCode);
    char *optionString = EncodeOptionBits(optionBits);

    printf("%s    (%s = 0x%5X)\n", pretty, optionString, optionBits);

    free(optionString);
    free(pretty);
    free(licenseCode);
  }

  puts("");
  return 0;
}
