
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include "openssl/rc5.h"
#include "miracl/miracl.h"
#include "rigup.h"
              // "ARBC3FGHJK96NEQDSMPVWXLZ2U45T87Y"
char charMap[] = "ARBC3FGHJK96NEQDSMPVWXLZ2U45T87Y";
char EncodeChar(uint32_t value)
{
  return charMap[value & 0x1f];
}


uint64_t EncodeSignature(uint64_t sig)
{
  assert( sig <= 0x0fffffffffffffff );
  return (sig << 4) | 0x00;
}

uint8_t* EncodeLicenseCodeInternal(const LicenseData *licenseData, const KeyData *keydata)
{
  if (!licenseData || !keydata)
    return 0;

  uint32_t optionBits = licenseData->optionBits;
  uint64_t sig1 = licenseData->sig1;
  uint64_t sig2 = licenseData->sig2;

  if (optionBits > 0x000fffff || sig1 > 0x0fffffffffffffff || sig2 > 0x0fffffffffffffff)
    return 0;

  sig1 = EncodeSignature(sig1);
  sig2 = EncodeSignature(sig2);

  static uint8_t licenseBits[35];

  uint64_t block1 = (uint64_t)optionBits << 48 | (sig2 & 0x00ffffffffffff00) >> 8;;
  uint64_t block2 = sig1 << 8 | (sig2 & 0xff);
  licenseBits[32] = sig2 >> 56;
  licenseBits[33] = sig1 >> 56;
  licenseBits[34] = optionBits >> 16;

  uint8_t RC5Key1[16], RC5Key2[16];
  RC5_32_KEY RC5Key;

  ParseHex(RC5Key1, 16, keydata->RC5Key1);
  RC5_32_set_key(&RC5Key, 16, RC5Key1, 0x0C);
  RC5_32_ecb_encrypt((uint8_t*)&block1, (uint8_t*)&block1, &RC5Key, 1);

  ParseHex(RC5Key2, 16, keydata->RC5Key2);
  RC5_32_set_key(&RC5Key, 16, RC5Key2, 16);
  RC5_32_ecb_encrypt((uint8_t*)&block2, (uint8_t*)&block2, &RC5Key, 0);

  for(int i = 0; i < 16; i++)
  {
    licenseBits[i] = (block1 >> (i*4)) & 0x0f;
    licenseBits[i+16] = (block2 >> (i*4)) & 0x0f;
  }

  return licenseBits;
}

char * EncodeLicenseCodeInternal2(uint8_t* licenseBits) {
  char *licenseCode = malloc(LICENSE_CODE_LENGTH + 1);

  for(int i = 0, j = 0; i < LICENSE_CODE_LENGTH; i += 4, j += 5)
  {
    uint32_t bitBuffer =
    ((uint32_t)(licenseBits[j]) & 0x0f) << 16
  | ((uint32_t)(licenseBits[j+1]) & 0x0f) << 12
  | ((uint32_t)(licenseBits[j+2]) & 0x0f) << 8
  | ((uint32_t)(licenseBits[j+3]) & 0x0f) << 4
  | ((uint32_t)(licenseBits[j+4]) & 0x0f);

    licenseCode[i] = EncodeChar(bitBuffer >> 15);
    licenseCode[i+1] = EncodeChar(bitBuffer >> 10);
    licenseCode[i+2] = EncodeChar(bitBuffer >> 5);
    licenseCode[i+3] = EncodeChar(bitBuffer);
  }

  licenseCode[LICENSE_CODE_LENGTH] = 0;
  return licenseCode;
}

char * EncodeLicenseCode(const LicenseData *licenseData, const KeyData *keydata)
{
  uint8_t* licenseBits = EncodeLicenseCodeInternal(licenseData, keydata);

  return EncodeLicenseCodeInternal2(licenseBits);
}


uint32_t DecodeOptionBits(const char *optionString)
{
  assert( optionString );
  assert( strlen(optionString) == 4);

  static const uint32_t reverseMap[] = {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,9,26,24,29,28,6,31,0,0,0,0,0,0,
    0,17,1,2,25,4,5,30,7,0,8,12,10,11,3,0,13,23,15,16,21,18,19,20,0,22,14,0,0,0,0,0,
    0,17,1,2,25,4,5,30,7,0,8,12,10,11,3,0,13,23,15,16,21,18,19,20,0,22,14,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
  };

  uint32_t optionBits =
    (reverseMap[(unsigned int)optionString[0]] << 15) |
    (reverseMap[(unsigned int)optionString[1]] << 10) |
    (reverseMap[(unsigned int)optionString[2]] << 5) |
    reverseMap[(unsigned int)optionString[3]];

  return optionBits;
}


// char * DecodeSerialNumber(const char *serialNumber)
// {
//   if(strlen(serialNumber) < 4)
//     return 0;

//   uint32_t header = *((uint32_t*)serialNumber);

//   for(int i = 0; i < 16; i++)
//     header =
//       ((header >> 7) & 1) ^
//       ((header >> 14) & 1) ^
//       ((header >> 23) & 1) ^
//       ((header >> 29) & 1) ^
//       ((header >> 31) & 1) ^
//       ((header >> 1) | (header << 31));

//   char* result = strdup(serialNumber);
//   *((uint32_t*)result) = header;
//   return result;
// }


static char prime1[]  = "AEBF94CEE3E707";
static char prime2[]  = "AEBF94D5C6AA71";
static char curve_a[] = "2982";
static char curve_b[] = "3408";
static char point1[]  = "7A3E808599A525";
static char point2[]  = "28BE7FAFD2A052";
void CreateSignature(uint64_t *sig1, uint64_t *sig2, uint8_t signatureDataHash[20], const char *privateKey)
{
    const int k_offset = 0; // optionally change ecssign starting offset (changes lic1; makes different licenses)

    big hash = mirvar(0);
    bytes_to_big(20, (char *)signatureDataHash, hash);

    big a = mirvar(0);
    instr(a, curve_a);
    big b = mirvar(0);
    instr(b, curve_b);
    big p = mirvar(0);
    instr(p, prime1);
    big q = mirvar(0);
    instr(q, prime2);
    big Gx = mirvar(0);
    instr(Gx, point1);
    big Gy = mirvar(0);
    instr(Gy, point2);
    big d = mirvar(0);
    instr(d, (char *)privateKey);
    big k = mirvar(0);
    big r = mirvar(0);
    big s = mirvar(0);
    big k1 = mirvar(0);
    big zero = mirvar(0);

    big f1 = mirvar(17);
    big f2 = mirvar(53);
    big f3 = mirvar(905461);
    big f4 = mirvar(60291817);

    incr(k, k_offset, k);

    epoint *G = epoint_init();
    epoint *kG = epoint_init();
    ecurve_init(a, b, p, MR_PROJECTIVE);
    epoint_set(Gx, Gy, 0, G);

    for(;;) {
        incr(k, 1, k);

        if(divisible(k, f1) || divisible(k, f2) || divisible(k, f3) || divisible(k, f4))
            continue;

        ecurve_mult(k, G, kG);
        epoint_get(kG, r, r);
        divide(r, q, q);

        if(mr_compare(r, zero) == 0)
            continue;

        xgcd(k, q, k1, k1, k1);
        mad(d, r, hash, q, q, s);
        mad(s, k1, k1, q, q, s);

        if(!divisible(s, f1) && !divisible(s, f2) && !divisible(s, f3) && !divisible(s, f4))
            break;
    }

    uint64_t sig = 0;

    big_to_bytes(8, r, (char *)&sig, 1);
    *sig1 = ByteSwap64(sig);

    big_to_bytes(8, s, (char *)&sig, 1);
    *sig2 = ByteSwap64(sig);

    free(G);
    free(kG);

}

LicenseData CreateLicenseCodeInternal(uint32_t optionBits, const KeyData *keydata)
{
  char *optionString = EncodeOptionBits(optionBits);
  char *encodedSerialNumber = keydata->serialNumber;

  // printf("optionString=%s sn=%s\n", optionString, encodedSerialNumber);

  uint8_t XXTEAKey[16];
  ParseHex(XXTEAKey, 16, keydata->XXTEAKey);

  uint32_t optionStringLength = strlen(optionString);
  uint32_t encodedSerialNumberLength = strlen(encodedSerialNumber);

  uint32_t signatureDataLength = (encodedSerialNumberLength + optionStringLength + 16);
  uint8_t *signatureData = malloc(signatureDataLength);

  memset(signatureData, 0, signatureDataLength);
  memcpy(signatureData, encodedSerialNumber, encodedSerialNumberLength);
  memcpy(signatureData + encodedSerialNumberLength, optionString, optionStringLength);
  memcpy(signatureData + encodedSerialNumberLength + optionStringLength, XXTEAKey, 16);
  
  free(optionString);
  // free(encodedSerialNumber);


  // XXTEA((uint32_t*)signatureData, signatureDataLength >> 2, (uint32_t*)XXTEAKey);
  // printf("Hashing: '%s' (%d, %d)\n", signatureData, strlen(signatureData), signatureDataLength);

  uint8_t hash[20];
  sha sh;
  shs_init(&sh);
  for(int i = 0; i < signatureDataLength; i++)
    shs_process(&sh, signatureData[i]);
  shs_hash(&sh, (char*)hash);

  uint64_t sig1, sig2;
  CreateSignature(&sig1, &sig2, hash, keydata->privateKey);

  // printf("sig1: 0x%016llx\n", sig1);
  // printf("sig2: 0x%016llx\n", sig2);

  free(signatureData);

  LicenseData licenseData = { optionBits, sig1, sig2 };
  return licenseData;
}

char * CreateLicenseCode(uint32_t optionBits, const KeyData *keydata)
{
  LicenseData licenseData = CreateLicenseCodeInternal(optionBits, keydata);
  return EncodeLicenseCode(&licenseData, keydata);
}


