
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "miracl/miracl.h"
#include "pbl/pbl.h"
#include "rigup.h"



static int set_compare(const void *a, const void *b)
{
  return strcasecmp(*(char **)a, *(char **)b);
}


static void usage()
{
  puts("Usage: rigup search [KEYFILE] DUMPFILE");
  puts("");
  puts("Scans for licenses in a memory dump. If the optional");
  puts("keyfile is provided, validates and print license options");
  puts("");
  puts("    KEYFILE       optional keyfile to check licenses againtst");
  puts("    DUMPFILE      binary file containing memory dump");
  puts("");
  exit(0);
}


int main_scanlicense(int argc, char *argv[])
{
  const char *dump_filename = 0;;
  const char *keys_filename = 0;

  switch (argc)
  {
    case 2:
      dump_filename = argv[1];
      break;

    case 3:
      keys_filename = argv[1];
      dump_filename = argv[2];
      break;

    default:
      usage();
  }

  KeyData *keydata = 0;
  if (keys_filename)
  {
    keydata = LoadKeyData(keys_filename);
    if (!keydata)
    {
      fprintf(stderr, "Loading keyfile '%s' failed\n", keys_filename);
      return 1;
    }
  }

  size_t datalen;
  void *data = LoadBinary(&datalen, dump_filename);
  if (!data)
  {
    free(keydata);
    fprintf(stderr, "Loading memory dump '%s' failed\n", dump_filename);
    return 1;
  }

  PblSet *set = pblSetNewTreeSet();
  pblSetSetCompareFunction(set, set_compare);

  const char *license = data;
  while (license <= (const char *)data + datalen - (LICENSE_CODE_LENGTH + 1))
  {
    static char allowed[] = "ABCDEFGHJKLMNPQRSTUVWXYZ23456789";
    // license character
    int i;
    for (i = LICENSE_CODE_LENGTH-1; i >= 0; --i)
      if (!license[i] || !strchr(allowed, license[i]))
	break;
    if (i >= 0)
    {
      license += i + 1;
      continue;
    }

    if (license[LICENSE_CODE_LENGTH] != '\0')
    {
      license += 1;
      continue;
    }

    // Found
    if (pblSetAdd(set, (void *)license))
    {
      char *pretty = PrettyLicenseCode(license);

      if (!keydata)
	puts(pretty);
      else
      {
	LicenseData licenseData;
	DecodeLicenseCode(&licenseData, license, keydata);

	int verified = VerifyLicenseCode(license, keydata);
	char *optionString = EncodeOptionBits(licenseData.optionBits);

	if (verified)
	  printf("%s    (%s = 0x%05X)    Ok.\n", pretty, optionString, licenseData.optionBits);
	else
	  printf("%s                        Failed.\n", pretty);

      }
      free(pretty);
    }

    license += LICENSE_CODE_LENGTH + 1;
  }

  puts("");

  pblSetFree(set);
  free(data);
  free(keydata);
  return 0;
}
