
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include "rigup.h"


static void DescramblePublicKey(uint8_t bytes[8]);


static void write_field(FILE *fp, const char *name, const char *data)
{
  if (!strlen(data))
    return;

  char buf[32];
  sprintf(buf, "%s:", name);
  fprintf(fp, "%-16s%s\n", buf, data);
}


static void WriteKeyData(const KeyData* data, FILE* fp)
{
  write_field(fp, "RC5KEY1",   data->RC5Key1);
  write_field(fp, "RC5KEY2",   data->RC5Key2);
  write_field(fp, "XXTEAKEY",  data->XXTEAKey);
  write_field(fp, "PUBKEY",    data->publicKey);
  write_field(fp, "PRIVKEY",   data->privateKey);
  write_field(fp, "SERIAL",    data->serialNumber);
}


int SaveKeyData(const char *filename, const KeyData* data)
{
  FILE *fp = fopen(filename, "w+");
  if(!fp) {
    fprintf(stderr, "File %s open failed\n", filename);
    return 0;
  }

  WriteKeyData(data, fp);

  fclose(fp);
  return 1;
}


void PrintKeyData(const KeyData* data)
{
  WriteKeyData(data, stdout);
}

#define BUFLEN 128

static int parse_line(const char *line, const char *id, char *target, unsigned int hexlength)
{
  char key[BUFLEN], value[BUFLEN];
  sscanf(line, " %[A-Za-z0-9] : %[^\n\r\t ]", key, value);

  if (strcasecmp(id, key))
    return 0;

  if (strlen(value) == 0)
    return 0;

  if (strlen(value) >= KEYDATA_FIELDSIZE)	// including null terminator!
    return -1;

  if (hexlength && !IsValidHex(value, hexlength))
    return -1;

  strcpy(target, value);
  return 0;
}

KeyData * LoadKeyData(const char *filename)
{
  FILE *fp = fopen(filename, "r");
  if(!fp)
  {
    fprintf(stderr, "File %s open failed\n", filename);
    return 0;
  }

  KeyData *keydata = calloc(1, sizeof(KeyData));

  while (!feof(fp))
  {
    char line[BUFLEN];

    if (!fgets(line, BUFLEN, fp))
      break;

    int error;
    error  = parse_line(line, "RC5KEY1", keydata->RC5Key1, 32);
    error |= parse_line(line, "RC5KEY2", keydata->RC5Key2, 32);
    error |= parse_line(line, "XXTEAKEY", keydata->XXTEAKey, 32);
    error |= parse_line(line, "PUBKEY", keydata->publicKey, 16);
    error |= parse_line(line, "PRIVKEY", keydata->privateKey, 16);
    error |= parse_line(line, "SERIAL", keydata->serialNumber, 0);

    if (error)
    {
      fprintf(stderr, "Error parsing line: '%s'\n", line);
      free(keydata);
      return 0;
    }
  }

  fclose(fp);
  return keydata;
}


char * FormatHex(char *out, const uint8_t *bytes, const size_t len)
{
    static const char *hex = "0123456789ABCDEF";

    if (!out)
      out = malloc(2 * len + 1);	// two chars per byte + null terminator

    for(size_t i = 0; i < len; ++i)
    {
      out[2*i]    = hex[(bytes[i]>>4)&0xF];
      out[2*i+1]  = hex[(bytes[i])&0xF];
    }

    out[2*len] = '\0';
    return out;
}

// parses a hexstring into a byte buffer with a certain capacity
// returns the number of bytes parsed or 0 if buffer was too small
size_t ParseHex(uint8_t *buffer, const size_t len, const char *hexstring)
{
  size_t    num_bytes;
  char      tmp_hex[3] = {0, 0,'\0'};	  // helper buffer to store two "hex" chars

  // validates format und length
  if(!IsValidHex(hexstring,2*len)) {
    fprintf(stderr, "Invalid Hexstring given: %s\n", hexstring);
    return 0;
  }

  // must be even
  if(strlen(hexstring)%2 > 0) {
    fprintf(stderr, "Hexstring %s has an odd length of %lu\n", hexstring, strlen(hexstring));
    return 0;
  }

  num_bytes = strlen(hexstring)/2;

  for(size_t i=0;i<num_bytes;++i) {
    tmp_hex[0] = hexstring[2*i];
    tmp_hex[1] = hexstring[2*i+1];

    buffer[i] = (uint8_t)strtoul(tmp_hex, NULL, 16);

    if(buffer[i]==0 && strcmp(tmp_hex, "00") != 0) {
      //free(buffer);
      fprintf(stderr, "Invalid hex sequence: %s in %s\n", tmp_hex, hexstring);
      return 0;
    }
  }

  return num_bytes;
}


int IsValidHex(const char* hexstring, const size_t len)
{
  if(len > 0 && strlen(hexstring) != len)	// check optional length
    return 0;

  while(*hexstring)				// check hex alphabet
    if (!isxdigit(*hexstring++))
      return 0;

  return 1;
}


void * LoadBinary(size_t *length, const char *filename)
{
  FILE *fp = fopen(filename, "rb");
  if(!fp)
    goto load_error;

  if (fseek(fp, 0, SEEK_END) != 0)
    goto load_error;

  long datasize = ftell(fp);

  if (fseek(fp, 0, SEEK_SET) != 0)
    goto load_error;

  void *data = malloc(datasize);
  if (!data)
    goto load_error;

  if ( fread(data, datasize, 1, fp) != 1 )
  {
    free(data);
    goto load_error;
  }

  fclose(fp);

  if (length)
    *length = datasize;

  return data;

load_error:
  if (fp)
    fclose(fp);
  return 0;
}


KeyData* ScanKeys(const void *data, size_t datasize)
{
  /*
    Offset	Data
      0		02 00 84 00 10 00
      For mso1074z-s, use: 01 00 84 00 10 00
      6		<16 bytes of XXTEAKey>
     22		20 00
     24		<16 bytes of RC5Key1>
     40		<16 bytes of RC5Key2>
     56		08 00
     58		<8 bytes of bit-shuffled ECC public key>
     66		40 00
     68		<64 bytes of some ASCII-HEX data>
    132		<END>
  */

  const unsigned int sequenceSize = 6 + 16 + 2 + 2*16 + 2 + 8 + 2 + 64;
  // static const uint8_t seq_1_ref[] = {0x02, 0x00, 0x84, 0x00, 0x10, 0x00};
  static const uint8_t seq_1_ref[] = {0x01, 0x00, 0x84, 0x00, 0x10, 0x00};

  const uint8_t *seq = data;
  while ( seq <= (const uint8_t *)data + datasize - sequenceSize )
  {
    // 64 bytes ASCII-HEX at end
    int i;
    for (i = 63; i >= 0; --i)
      if (!isxdigit(seq[68 + i]))
	break;
    if (i >= 0)
    {
      seq += i + 1;
      continue;
    }

    // 6 bytes at beginning of sequence
    if (!memchr(seq_1_ref, seq[5], 6))
    {
      seq += 6;
      continue;
    }

    // 2 byte patterns
    if ( (seq[23]!=0x00 && seq[23]!=0x20) ||
	 (seq[57]!=0x00 && seq[57]!=0x08) ||
	 (seq[67]!=0x00 && seq[67]!=0x40) )
    {
      seq += 2;
      continue;
    }

    // complete compare...
    if ( memcmp(seq, seq_1_ref, 6) != 0 ||
	 seq[22]!=0x20 || seq[23]!=0x00 ||
	 seq[56]!=0x08 || seq[57]!=0x00 ||
	 seq[66]!=0x40 || seq[67]!=0x00 )
    {
      seq += 1;
      continue;
    }

    // Found!
    break;
  }

  if (seq >= (const uint8_t *)data + datasize - sequenceSize)
    return 0;

  KeyData *keydata = calloc(1, sizeof(KeyData));

  FormatHex(keydata->XXTEAKey, seq + 6, 16);
  FormatHex(keydata->RC5Key1, seq + 24, 16);
  FormatHex(keydata->RC5Key2, seq + 40, 16);

  uint8_t publicKey[8];
  memcpy(publicKey, seq + 58, 8);
  DescramblePublicKey(publicKey);
  FormatHex(keydata->publicKey, publicKey, 8);

  // Scan for serial number: DS2x123456789\0
  // Changed for MSO1074Z-S: DS1ZD123456789\0
  //const unsigned int serialNumberLength = 3 + 1 + 9 + 1;
  const unsigned int serialNumberLength = 3 + 2 + 9 + 1;
  const char *serialNumber = data;
  while ( serialNumber <= (const char *)data + datasize - serialNumberLength )
  {
    // 9 digits at end
    int i;
    for (i = 8; i >= 0; --i)
      if (!isdigit(serialNumber[5 + i]))
	break;
    if (i >= 0)
    {
      serialNumber += i + 1;
      continue;
    }

    // 5 chars beginning
    if ( serialNumber[4]!='D' && serialNumber[3]!='Z' && serialNumber[2]!='1' && serialNumber[1]!='S' && serialNumber[0]!='D' )
    {
      serialNumber += 5;
      continue;
    }

    // complete compare
    if ( serialNumber[0]!='D' || serialNumber[1]!='S' || serialNumber[2]!='1' ||
         !isalpha(serialNumber[3]) || !isalnum(serialNumber[4]) || serialNumber[14]!='\0' )
    {
      serialNumber += 1;
      continue;
    }

    // Found!
    break;
  }

  if (serialNumber <= (const char *)data + datasize - serialNumberLength)
    strcpy(keydata->serialNumber, (char *)serialNumber);

  return keydata;
}


 uint32_t unshuffle(uint32_t x)
{
    x = ((x & 0x22222222) << 1) | ((x >> 1) & 0x22222222) | (x & 0x99999999);
    x = ((x & 0x0C0C0C0C) << 2) | ((x >> 2) & 0x0C0C0C0C) | (x & 0xC3C3C3C3);
    x = ((x & 0x00F000F0) << 4) | ((x >> 4) & 0x00F000F0) | (x & 0xF00FF00F);
    x = ((x & 0x0000FF00) << 8) | ((x >> 8) & 0x0000FF00) | (x & 0xFF0000FF);
    return x;
}

static void DescramblePublicKey(uint8_t bytes[8])
{
  uint32_t word[2];

  word[0] =
    (uint32_t)bytes[3] << 24 |
    (uint32_t)bytes[2] << 16 |
    (uint32_t)bytes[1] << 8 |
    (uint32_t)bytes[0];

  word[1] =
    (uint32_t)bytes[7] << 24 |
    (uint32_t)bytes[6] << 16 |
    (uint32_t)bytes[5] << 8 |
    (uint32_t)bytes[4];

  word[0] = unshuffle(word[0]);
  word[1] = unshuffle(word[1]);

  bytes[7] = word[0];
  bytes[6] = word[0] >> 8;
  bytes[5] = word[0] >> 16;
  bytes[4] = word[0] >> 24;
  bytes[3] = word[1];
  bytes[2] = word[1] >> 8;
  bytes[1] = word[1] >> 16;
  bytes[0] = word[1] >> 24;
}

char * PrettyLicenseCode(const char *licenseCode)
{
  if (!licenseCode)
    return 0;

  const unsigned int chunksize = 7;
  const unsigned int numchunks = (strlen(licenseCode) + chunksize - 1) / chunksize;
  char *pretty = calloc(1, LICENSE_CODE_LENGTH + numchunks);
  pretty[0] = '\0';

  unsigned int i;
  for (i = 0; i < numchunks; ++i)
  {
    strncpy(pretty + i * (chunksize+1), licenseCode + (i * chunksize), chunksize);
    pretty[chunksize + i * (chunksize+1)] = '-';
  }

  pretty[LICENSE_CODE_LENGTH + numchunks - 1] = '\0';	// Replace trailing '-' with null terminator
  return pretty;
}


char * CompactLicenseCode(const char *licenseCode)
{
  static const char allowed[] = "ABCDEFGHJKLMNPQRSTUVWXYZ23456789";

  if (!licenseCode)
    return 0;

  char *compact = malloc(LICENSE_CODE_LENGTH + 1);

  int i = 0;
  const char *p = licenseCode;

  while (i < LICENSE_CODE_LENGTH && *p)
  {
    char c = *p++;

    if (!isalnum(c))
      continue;

    if (!strchr(allowed, c))
      break;

    compact[i++] = toupper(c);
  }

  if (i < LICENSE_CODE_LENGTH || *p)
  {
    free(compact);
    return 0;
  }

  compact[LICENSE_CODE_LENGTH] = '\0';
  return compact;
}

